import * as state from './state.js';

export const backgroundMusic = new Audio();
export const attractMusicPlayer = new Audio();
export const navigateSound = new Audio();
export const confirmSound = new Audio();
export const cancelSound = new Audio();

let musicFadeIntervals = new Map();

export function t(key, replacements = {}) {
    let text = state.currentTranslations[key] || key;
    for (const [placeholder, value] of Object.entries(replacements)) {
        text = text.replace(`{${placeholder}}`, value);
    }
    return text;
}


export function applyTranslationsToDOM(translations) {
    
    if (translations) {
        state.setCurrentTranslations(translations);
    }
    document.querySelectorAll('[data-i18n-key]').forEach(el => {
        const key = el.dataset.i18nKey;
        const translation = t(key);
        if (el.tagName === 'INPUT' || el.tagName === 'TEXTAREA') {
            if (el.placeholder) el.placeholder = translation;
        } else if (key.includes('tooltip')) {
            el.innerHTML = translation;
        } else {
            el.textContent = translation;
        }
    });
}

export function playSound(sound) {
    sound.currentTime = 0;
    sound.play().catch(e => console.error("Sound effect failed to play:", e));
}

export function fadeOutAndStop(audioPlayer) {
    if (musicFadeIntervals.has(audioPlayer)) {
        clearInterval(musicFadeIntervals.get(audioPlayer));
    }
    if (audioPlayer.paused || audioPlayer.volume === 0) return;

    const initialVolume = audioPlayer.volume;
    const fadeInterval = setInterval(() => {
        const newVolume = audioPlayer.volume - 0.05;
        if (newVolume <= 0) {
            audioPlayer.pause();
            audioPlayer.currentTime = 0;
            audioPlayer.volume = initialVolume;
            clearInterval(fadeInterval);
            musicFadeIntervals.delete(audioPlayer);
        } else {
            audioPlayer.volume = newVolume;
        }
    }, 25);
    musicFadeIntervals.set(audioPlayer, fadeInterval);
}

export function anyModalOpen() {
    return state.isStartMenuOpen || state.isSystemOptionsMenuOpen || state.isIngameOptionsOpen ||
           state.isGeneralOptionsOpen || state.isNetworkFeaturesOpen ||
           state.isPowerMenuOpen || state.isSaveStatesMenuOpen || state.isNetplayUsernameOpen ||
           state.isRaLoginOpen || state.isLobbyOpen;
}

export function shuffleArray(array) {
    for (let i = array.length - 1; i > 0; i--) {
        const j = Math.floor(Math.random() * (i + 1));
        [array[i], array[j]] = [array[j], array[i]];
    }
    return array;
}